/**
 * Utilidad para manejar drag & drop de etapas
 * Permite reorganizar etapas arrastrándolas y soltándolas
 */

class StageDragDrop {
    constructor(projectId, onMoveCallback) {
        this.projectId = projectId;
        this.onMoveCallback = onMoveCallback;
        this.draggedElement = null;
        this.draggedStageId = null;
    }

    /**
     * Inicializa los eventos de drag & drop en los elementos de etapa
     */
    init() {
        const stageElements = document.querySelectorAll('[data-stage-id]');
        
        stageElements.forEach(element => {
            element.setAttribute('draggable', 'true');
            
            // Eventos de arrastre
            element.addEventListener('dragstart', this.handleDragStart.bind(this));
            element.addEventListener('dragend', this.handleDragEnd.bind(this));
            
            // Eventos de zona de soltar
            element.addEventListener('dragover', this.handleDragOver.bind(this));
            element.addEventListener('drop', this.handleDrop.bind(this));
            element.addEventListener('dragleave', this.handleDragLeave.bind(this));
        });
    }

    handleDragStart(e) {
        this.draggedElement = e.target;
        this.draggedStageId = e.target.getAttribute('data-stage-id');
        
        e.target.classList.add('dragging');
        e.dataTransfer.effectAllowed = 'move';
        e.dataTransfer.setData('text/html', e.target.innerHTML);
    }

    handleDragEnd(e) {
        e.target.classList.remove('dragging');
        
        // Remover clases de hover de todos los elementos
        document.querySelectorAll('.drag-over').forEach(el => {
            el.classList.remove('drag-over');
        });
    }

    handleDragOver(e) {
        if (e.preventDefault) {
            e.preventDefault();
        }
        
        e.dataTransfer.dropEffect = 'move';
        
        const targetElement = e.target.closest('[data-stage-id]');
        if (targetElement && targetElement !== this.draggedElement) {
            targetElement.classList.add('drag-over');
        }
        
        return false;
    }

    handleDragLeave(e) {
        const targetElement = e.target.closest('[data-stage-id]');
        if (targetElement) {
            targetElement.classList.remove('drag-over');
        }
    }

    async handleDrop(e) {
        if (e.stopPropagation) {
            e.stopPropagation();
        }

        const targetElement = e.target.closest('[data-stage-id]');
        
        if (targetElement && this.draggedElement !== targetElement) {
            const targetStageId = targetElement.getAttribute('data-stage-id');
            const newParentId = targetElement.getAttribute('data-parent-id');
            
            try {
                // Llamar al API para mover la etapa
                await this.moveStage(this.draggedStageId, newParentId);
                
                // Callback para actualizar la UI
                if (this.onMoveCallback) {
                    this.onMoveCallback(this.draggedStageId, newParentId);
                }
                
                console.log(`Etapa ${this.draggedStageId} movida bajo ${newParentId || 'raíz'}`);
            } catch (error) {
                console.error('Error al mover la etapa:', error);
                alert('No se pudo mover la etapa: ' + error.message);
            }
        }

        targetElement?.classList.remove('drag-over');
        return false;
    }

    /**
     * Llama al API para mover una etapa
     */
    async moveStage(stageId, newParentId) {
        const response = await fetch(`/stage/${stageId}/move`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                parent_id: newParentId,
                project_id: this.projectId
            })
        });

        const data = await response.json();

        if (!response.ok || !data.success) {
            throw new Error(data.message || 'Error al mover la etapa');
        }

        return data;
    }
}

/**
 * Funciones para exportar y guardar la estructura
 */
async function exportProjectStructure(projectId) {
    try {
        const response = await fetch(`/project/${projectId}/stages/export`);
        const data = await response.json();

        if (data.success) {
            // Descargar como archivo JSON
            const blob = new Blob([JSON.stringify(data.data, null, 2)], { type: 'application/json' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `estructura-proyecto-${projectId}.json`;
            a.click();
            URL.revokeObjectURL(url);
            
            console.log('Estructura exportada correctamente');
        } else {
            throw new Error(data.message);
        }
    } catch (error) {
        console.error('Error al exportar estructura:', error);
        alert('Error al exportar la estructura: ' + error.message);
    }
}

async function saveAsDefaultStructure(projectId) {
    if (!confirm('¿Deseas guardar esta estructura como plantilla por defecto para nuevos proyectos?')) {
        return;
    }

    try {
        const response = await fetch(`/project/${projectId}/stages/save-default`, {
            method: 'POST'
        });

        const data = await response.json();

        if (data.success) {
            alert('Estructura guardada como plantilla por defecto exitosamente');
            console.log('Estructura guardada en ProjectDefaultStructure.json');
        } else {
            throw new Error(data.message);
        }
    } catch (error) {
        console.error('Error al guardar estructura:', error);
        alert('Error al guardar la estructura: ' + error.message);
    }
}

// Ejemplo de uso:
// const dragDrop = new StageDragDrop(projectId, (stageId, newParentId) => {
//     // Actualizar la UI después de mover
//     location.reload(); // o actualizar dinámicamente
// });
// dragDrop.init();
