﻿import Modal from "../../../../components/modal/modal.js";

// Modal para subir evidencias/documentos a una etapa
export const modalUploadEvidence = new Modal({
    title: "Subir documento",
    subtitle: "Añade un documento o evidencia a la etapa seleccionada.",
    content: ({id, name}) => `
        <div class="pt-2 rounded-md">
            <form id="formUploadEvidence" class="flex flex-col gap-2">
                <div class="flex flex-col gap-2">
                    <label for="name_document" class="text-sm font-medium text-zinc-700">
                        Nombre del documento <span class="text-red-500">*</span>
                    </label>
                    <input 
                        type="text" 
                        id="name_document" 
                        name="name_document"
                        placeholder="Ej: Documento de planeación"
                        required
                        class="text-zinc-500 p-2 border border-zinc-200 rounded-lg bg-zinc-50 w-full outline-none focus:bg-zinc-100 transition-all ease-in-out"
                    />
                </div>

                <div class="flex flex-col gap-2">
                    <label for="description_document" class="text-sm font-medium text-zinc-500">
                        Descripción
                    </label>
                    <textarea 
                        id="description_document" 
                        name="description_document"
                        rows="3"
                        placeholder="Ej: Este documento representa la planificación del proyecto..."
                        class="text-zinc-700 p-2 border border-zinc-200 rounded-lg bg-zinc-50 w-full outline-none focus:bg-zinc-100 transition-all ease-in-out"
                    ></textarea>
                </div>

                <div class="flex flex-col gap-2">
                    <label class="text-sm font-medium text-zinc-500">
                        Archivo <span class="text-red-500">*</span>
                    </label>
                    <div id="dropZone" class="border-2 border-dashed border-zinc-300 rounded-lg p-6 text-center hover:border-blue-400 transition-colors cursor-pointer bg-zinc-50">
                        <input 
                            type="file" 
                            id="documentInput" 
                            name="document"
                            accept=".pdf,.docx,.xlsx,.png,.jpg,.jpeg,.pptx,.txt"
                            required
                            class="hidden"
                        />
                        <div id="dropZoneContent">
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" class="w-10 h-10 mx-auto text-zinc-400 mb-2">
                                <path fill-rule="evenodd" d="M11.47 2.47a.75.75 0 0 1 1.06 0l4.5 4.5a.75.75 0 0 1-1.06 1.06l-3.22-3.22V16.5a.75.75 0 0 1-1.5 0V4.81L8.03 8.03a.75.75 0 0 1-1.06-1.06l4.5-4.5ZM3 15.75a.75.75 0 0 1 .75.75v2.25a1.5 1.5 0 0 0 1.5 1.5h13.5a1.5 1.5 0 0 0 1.5-1.5V16.5a.75.75 0 0 1 1.5 0v2.25a3 3 0 0 1-3 3H5.25a3 3 0 0 1-3-3V16.5a.75.75 0 0 1 .75-.75Z" clip-rule="evenodd" />
                            </svg>
                            <p class="text-zinc-600 font-medium mb-1 text-xs">Haz clic para seleccionar o arrastra el archivo aquí</p>
                            
                        </div>
                        <div id="filePreview" class="hidden">
                            <div class="flex items-center justify-center gap-3">
                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" class="w-8 h-8 text-blue-500 flex-shrink-0">
                                    <path fill-rule="evenodd" d="M5.625 1.5c-1.036 0-1.875.84-1.875 1.875v17.25c0 1.035.84 1.875 1.875 1.875h12.75c1.035 0 1.875-.84 1.875-1.875V12.75A3.75 3.75 0 0 0 16.5 9h-1.875a1.875 1.875 0 0 1-1.875-1.875V5.25A3.75 3.75 0 0 0 9 1.5H5.625ZM7.5 15a.75.75 0 0 1 .75-.75h7.5a.75.75 0 0 1 0 1.5h-7.5A.75.75 0 0 1 7.5 15Zm.75 2.25a.75.75 0 0 0 0 1.5H12a.75.75 0 0 0 0-1.5H8.25Z" clip-rule="evenodd" />
                                    <path d="M12.971 1.816A5.23 5.23 0 0 1 14.25 5.25v1.875c0 .207.168.375.375.375H16.5a5.23 5.23 0 0 1 3.434 1.279 9.768 9.768 0 0 0-6.963-6.963Z" />
                                </svg>
                                <div class="text-left flex-1">
                                    <p id="fileName" class="text-sm font-medium text-zinc-700 truncate"></p>
                                    <p id="fileSize" class="text-xs text-zinc-500"></p>
                                </div>
                                <button type="button" id="removeFile" class="text-red-500 hover:text-red-700 flex-shrink-0">
                                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5">
                                        <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12" />
                                    </svg>
                                </button>
                            </div>
                        </div>
                    </div>
                    <p class="text-xs text-zinc-500 text-end w-full">PDF, DOCX, XLSX, PNG, JPG, PPTX, TXT (Máx. 5MB)</p>
                </div>

                <input type="hidden" id="stage_id" name="stage_id" value="${id}" />

                <div class="flex justify-end gap-3 pt-4 border-t border-zinc-200">
                    <button type="submit" id="submitUpload" class="px-6 py-2 bg-zinc-900 text-white rounded-lg hover:bg-zinc-950 transition-colors flex items-center gap-2">
                        <span>Subir documento</span>
                    </button>
                </div>
            </form>
        </div>`,
    onOpen: (modal) => {
        const dropZone = modal.querySelector('#dropZone');
        const fileInput = modal.querySelector('#documentInput');
        const dropZoneContent = modal.querySelector('#dropZoneContent');
        const filePreview = modal.querySelector('#filePreview');
        const removeFileBtn = modal.querySelector('#removeFile');

        // Formatear tamaño de archivo
        const formatFileSize = (bytes) => {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
        };

        // Mostrar preview del archivo
        const showFilePreview = (file) => {
            if (file.size > 5 * 1024 * 1024) {
                alert('El archivo es demasiado grande. Tamaño máximo: 5MB');
                fileInput.value = '';
                return;
            }

            dropZoneContent.classList.add('hidden');
            filePreview.classList.remove('hidden');
            modal.querySelector('#fileName').textContent = file.name;
            modal.querySelector('#fileSize').textContent = formatFileSize(file.size);
        };

        // Click en la zona para abrir selector
        dropZone?.addEventListener('click', () => fileInput.click());

        // Drag & Drop
        dropZone?.addEventListener('dragover', (e) => {
            e.preventDefault();
            dropZone.classList.add('border-blue-500', 'bg-blue-50');
        });

        dropZone?.addEventListener('dragleave', () => {
            dropZone.classList.remove('border-blue-500', 'bg-blue-50');
        });

        dropZone?.addEventListener('drop', (e) => {
            e.preventDefault();
            dropZone.classList.remove('border-blue-500', 'bg-blue-50');
            
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                fileInput.files = files;
                showFilePreview(files[0]);
            }
        });

        // Selección de archivo
        fileInput?.addEventListener('change', (e) => {
            const file = e.target.files[0];
            if (file) {
                showFilePreview(file);
            }
        });

        // Remover archivo
        removeFileBtn?.addEventListener('click', (e) => {
            e.stopPropagation();
            fileInput.value = '';
            filePreview.classList.add('hidden');
            dropZoneContent.classList.remove('hidden');
        });
    }
});
