/**
 * Controlador de la página de Administración
 * Gestiona la configuración de la estructura por defecto de proyectos
 */

class AdministrationPage {
    constructor() {
        this.tree = null;
        this.service = null;
        this.structure = null;
    }

    /**
     * Inicializa la página
     */
    async init() {
        try {
            // Inicializar servicios
            this.service = new ProjectStructureService();
            
            // Cargar estructura desde el elemento en el DOM
            await this.loadStructure();
            
            // Inicializar árbol
            this.tree = new ProjectStructureTree('projectTree', this.structure);
            this.tree.render();
            
            // Configurar manejadores de eventos
            this.setupEventHandlers();
            
            console.log('Página de administración inicializada correctamente');
        } catch (error) {
            console.error('Error al inicializar página:', error);
            this.showError('Error al cargar la página de administración');
        }
    }

    /**
     * Carga la estructura desde el elemento script en el DOM
     */
    async loadStructure() {
        try {
            const structureElement = document.getElementById('defaultStructure');
            if (structureElement) {
                this.structure = JSON.parse(structureElement.textContent);
                console.log('Estructura cargada:', this.structure);
            } else {
                throw new Error('No se encontró el elemento de estructura');
            }
        } catch (error) {
            console.error('Error al cargar estructura:', error);
            throw error;
        }
    }

    /**
     * Configura los manejadores de eventos
     */
    setupEventHandlers() {
        // Botón guardar estructura
        const saveBtn = document.getElementById('saveStructureBtn');
        if (saveBtn) {
            saveBtn.addEventListener('click', () => this.saveStructure());
        }

        // Botón agregar etapa raíz
        const addRootBtn = document.getElementById('addRootStageBtn');
        if (addRootBtn) {
            addRootBtn.addEventListener('click', () => this.addRootStage());
        }

        // Escuchar cambios en el árbol
        document.addEventListener('tree:nodeAdded', (e) => {
            console.log('Nodo agregado:', e.detail);
        });

        document.addEventListener('tree:nodeDeleted', (e) => {
            console.log('Nodo eliminado:', e.detail);
        });

        document.addEventListener('tree:nodeMoved', (e) => {
            console.log('Nodo movido:', e.detail);
        });
    }

    /**
     * Guarda la estructura actual
     */
    async saveStructure() {
        try {
            const saveBtn = document.getElementById('saveStructureBtn');
            if (saveBtn) {
                saveBtn.disabled = true;
                saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Guardando...';
            }

            // Obtener estructura actual del árbol
            const currentStructure = this.tree.getStructure();
            
            // Validar estructura
            if (!this.service.validateStructure(currentStructure)) {
                throw new Error('La estructura no es válida');
            }

            // Guardar en el servidor
            const response = await this.service.saveStructure(currentStructure);

            if (response.success) {
                this.showSuccess('Estructura guardada correctamente');
                
                // Actualizar estructura local
                this.structure = currentStructure;
            } else {
                throw new Error(response.message || 'Error al guardar estructura');
            }
        } catch (error) {
            console.error('Error al guardar estructura:', error);
            this.showError(error.message || 'Error al guardar la estructura');
        } finally {
            const saveBtn = document.getElementById('saveStructureBtn');
            if (saveBtn) {
                saveBtn.disabled = false;
                saveBtn.innerHTML = '<i class="fas fa-save mr-2"></i>Guardar Estructura';
            }
        }
    }

    /**
     * Agrega una nueva etapa raíz
     */
    addRootStage() {
        const name = prompt('Nombre de la nueva etapa:');
        if (name && name.trim()) {
            this.tree.addNode(null, name.trim());
            this.showSuccess('Etapa agregada correctamente');
        }
    }

    /**
     * Muestra un mensaje de éxito
     */
    showSuccess(message) {
        this.showNotification(message, 'success');
    }

    /**
     * Muestra un mensaje de error
     */
    showError(message) {
        this.showNotification(message, 'error');
    }

    /**
     * Muestra una notificación
     */
    showNotification(message, type = 'info') {
        // Crear elemento de notificación
        const notification = document.createElement('div');
        notification.className = `fixed top-4 right-4 px-6 py-3 rounded-lg shadow-lg z-50 ${
            type === 'success' ? 'bg-green-500' : 
            type === 'error' ? 'bg-red-500' : 
            'bg-blue-500'
        } text-white`;
        notification.textContent = message;

        // Agregar al DOM
        document.body.appendChild(notification);

        // Remover después de 3 segundos
        setTimeout(() => {
            notification.classList.add('opacity-0', 'transition-opacity', 'duration-300');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }
}

// Inicializar cuando el DOM esté listo
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        const page = new AdministrationPage();
        page.init();
    });
} else {
    const page = new AdministrationPage();
    page.init();
}
