// js/components/Modal.js
export default class Modal {
  constructor({ title = "", subtitle = "", content = "" }) {
    this.title = title;
    this.subtitle = subtitle;
    this.content = content;
    this.modal = null;
    this.modalContent = null;
    this.handleEsc = this.handleEsc.bind(this);
  }

  build() {
    // Overlay
    this.modal = document.createElement("div");
    this.modal.className = "fixed inset-0 flex items-center justify-center bg-black/10 backdrop-blur-[2px] z-50 opacity-0 transition-opacity duration-300";

    // Contenido
    this.modalContent = document.createElement("div");
    this.modalContent.className = "relative bg-white text-white p-5 rounded-lg max-w-lg w-11/12 shadow-lg transform scale-95 transition-transform duration-300";

    // Botón de cerrar (X)
    const closeBtn = document.createElement("button");
    closeBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="size-7 p-1"><path stroke-linecap="round" stroke-linejoin="round" d="M6 18 18 6M6 6l12 12" /></svg>`;
    closeBtn.className = "absolute top-3 right-3 text-2xl text-gray-400 hover:bg-zinc-100 rounded-md hover:text-zinc-600 transition cursor-pointer transition-all ease-in-out hover:scale-110";
    closeBtn.addEventListener("click", () => this.close());

    this.modalContent.appendChild(closeBtn);

    // Título
    if (this.title) {
      const h2 = document.createElement("h2");
      h2.textContent = this.title;
      h2.className = "text-xl font-bold text-zinc-800";
      this.modalContent.appendChild(h2);
    }

    // Subtítulo
    if (this.subtitle) {
      const h3 = document.createElement("h3");
      h3.textContent = this.subtitle;
      h3.className = "text-zinc-400 text-sm pb-2 border-b-zinc-200 font-medium border-b";
      this.modalContent.appendChild(h3);
    }

    // Contenido
    const contentDiv = document.createElement("div");
    if (typeof this.content === "string") {
      contentDiv.innerHTML = this.content;
    } else {
      contentDiv.appendChild(this.content);
    }
    this.modalContent.appendChild(contentDiv);

    this.modal.appendChild(this.modalContent);
  }

  open() {
    this.build();
    document.body.appendChild(this.modal);

    // Listener ESC
    document.addEventListener("keydown", this.handleEsc);

    // Animación entrada
    requestAnimationFrame(() => {
      this.modal.classList.remove("opacity-0");
      this.modal.classList.add("opacity-100");
      this.modalContent.classList.remove("scale-95");
      this.modalContent.classList.add("scale-100");
    });
  }

  close() {
    this.modal.classList.remove("opacity-100");
    this.modal.classList.add("opacity-0");
    this.modalContent.classList.remove("scale-100");
    this.modalContent.classList.add("scale-95");

    setTimeout(() => {
      if (this.modal && document.body.contains(this.modal)) {
        document.body.removeChild(this.modal);
      }
      document.removeEventListener("keydown", this.handleEsc);
    }, 300);
  }

  handleEsc(e) {
    if (e.key === "Escape") {
      this.close();
    }
  }
}
