export default class ModalBar {
  constructor({ 
    title = "", 
    content = "", 
    side = "right", 
    width = null,
    onOpen = null,
    onClose = null 
  }) {
    this.title = title;
    this.content = content; // string o función
    this.side = side;       // "left" o "right"
    this.width = width;     // ancho del modal bar (opcional)
    this.modal = null;
    this.modalContent = null;
    this.onOpen = onOpen;
    this.onClose = onClose;
    this.handleEsc = this.handleEsc.bind(this);
  }

  build(props = {}) {
    // Overlay
    this.modal = document.createElement("div");
    this.modal.className = "fixed inset-0 bg-black/20 z-50 opacity-0 transition-opacity duration-300";

    // Panel lateral
    this.modalContent = document.createElement("div");
    this.modalContent.className = `
      fixed top-0 ${this.side}-0 h-full bg-white shadow-lg transform transition-transform duration-300
      xs:w-10/12 sm:w-7/12 md:w-6/12 lg:w-4/12 xl:w-4/12 2xl:w-3/12
    `;

    if (this.width) {
      this.modalContent.style.width = this.width;
    }

    this.modalContent.style[this.side] = "0";
    this.modalContent.style.transform = `translateX(${this.side === "right" ? "100%" : "-100%"})`;

    // Evitar que clic dentro cierre el modal
    this.modalContent.addEventListener("click", (e) => e.stopPropagation());

    // Header
    const header = document.createElement("div");
    header.className = "p-5 flex justify-between items-center relative";

    const titleEl = document.createElement("h2");
    titleEl.textContent = this.title;
    titleEl.className = "text-xl font-bold text-zinc-800";
    header.appendChild(titleEl);

    const closeBtn = document.createElement("button");
    closeBtn.innerHTML = `
      <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" 
           stroke-width="2" stroke="currentColor" class="size-7 p-1">
        <path stroke-linecap="round" stroke-linejoin="round" d="M6 18 18 6M6 6l12 12"/>
      </svg>`;
    closeBtn.className =
      "absolute top-3 right-3 text-2xl text-gray-400 hover:bg-zinc-100 rounded-md hover:text-zinc-600 transition cursor-pointer transition-all ease-in-out hover:scale-110";
    closeBtn.addEventListener("click", () => this.close());
    header.appendChild(closeBtn);

    this.modalContent.appendChild(header);

    // Body
    const contentDiv = document.createElement("div");
    contentDiv.className = "px-5 overflow-y-auto h-[calc(100%-80px)]";

    if (typeof this.content === "function") {
      contentDiv.innerHTML = this.content(props);
    } else if (typeof this.content === "string") {
      contentDiv.innerHTML = this.content;
    } else {
      contentDiv.appendChild(this.content);
    }

    this.modalContent.appendChild(contentDiv);
    this.modal.appendChild(this.modalContent);
  }

  open(props = {}) {
    this.build(props);
    document.body.appendChild(this.modal);
    document.addEventListener("keydown", this.handleEsc);

    // Forzar reflow
    void this.modal.offsetWidth;

    // Animar entrada
    this.modal.classList.remove("opacity-0");
    this.modal.classList.add("opacity-100");
    this.modalContent.style.transform = "translateX(0)";

    // Esperar fin de animación antes de disparar evento/callback
    setTimeout(() => {
      if (typeof this.onOpen === "function") this.onOpen(this.modal);
      this.modal.dispatchEvent(new CustomEvent("modalbar:open", { detail: { modal: this } }));
    }, 300);
  }

  close() {
    this.modal.classList.remove("opacity-100");
    this.modal.classList.add("opacity-0");
    this.modalContent.style.transform = `translateX(${this.side === "right" ? "100%" : "-100%"})`;

    setTimeout(() => {
      if (this.modal && document.body.contains(this.modal)) {
        document.body.removeChild(this.modal);
      }
      document.removeEventListener("keydown", this.handleEsc);

      // Disparar evento/callback
      if (typeof this.onClose === "function") this.onClose();
      this.modal.dispatchEvent(new CustomEvent("modalbar:close"));
    }, 300);
  }

  handleEsc(e) {
    if (e.key === "Escape") this.close();
  }
}
