<?php

namespace App\Repositories;

use App\Config\ContextDB;
use App\Models\Document;
use App\Models\User;
use PDO;

class EvidenceRepository {
    private PDO $context;

    public function __construct(){
        $this->context = (new ContextDB())->conn;
    }

     public function findById(int $documentId): ?Document
        {
            $consult = $this->context->prepare("
                SELECT id, stage_id, file_name, file_path, uploaded_by
                FROM evidence
                WHERE id = :document_id
                LIMIT 1
            ");

            $consult->execute([
                "document_id" => $documentId
            ]);

            $data = $consult->fetch(PDO::FETCH_ASSOC);
            return $data ? new Document($data) : null;
        }

    public function submit(Document $document)
    {
        $consult = $this->context->prepare("
            INSERT INTO evidence(stage_id, file_name, file_path, uploaded_by, name_document, description_document) 
            VALUES(:stage_id, :file_name, :file_path, :uploaded_by, :name_document, :description_document)
        ");

        $consult->execute([
            "stage_id" => $document->stage_id,
            "file_name" => $document->file_name,
            "file_path" => $document->file_path,
            "uploaded_by" => $document->uploaded_by,
            "name_document" => $document->name_document,
            "description_document" => $document->description_document
        ]);

        if ($consult->rowCount() > 0) {
            $documentId = $this->context->lastInsertId();
            return $this->findById($documentId);
        }

        return null;
    }

    public function findByStageId(int $stageId): array
    {
        $consult = $this->context->prepare("
            SELECT 
                e.id, 
                e.stage_id, 
                e.file_name, 
                e.file_path, 
                e.created_at,
                e.uploaded_by,
                e.name_document,
                e.description_document,
                u.id AS user_id,
                u.username AS user_name
            FROM evidence e
            LEFT JOIN users u ON u.id = e.uploaded_by
            WHERE e.stage_id = :stage_id
            ORDER BY e.created_at DESC
        ");

        $consult->execute([
            "stage_id" => $stageId
        ]);

        $results = $consult->fetchAll(PDO::FETCH_ASSOC);

        return array_map(function($row) {
            $document = new Document([
                'id' => $row['id'],
                'stage_id' => $row['stage_id'],
                'file_name' => $row['file_name'],
                'file_path' => $row['file_path'],
                'uploaded_by' => $row['uploaded_by'] ?? null,
                'created_at' => $row['created_at'] ?? null,
                'name_document' => $row['name_document'] ?? '',
                'description_document' => $row['description_document'] ?? ''
            ]);

            if (!empty($row['user_id'])) {
                $document->user = new User([
                    'id' => $row['user_id'],
                    'username' => $row['user_name']
                ]);
            }

            return $document;
        }, $results);
    }

    public function delete(int $documentId): bool
    {
        $consult = $this->context->prepare("
            DELETE FROM evidence
            WHERE id = :document_id
        ");

        $consult->execute([
            "document_id" => $documentId
        ]);

        return $consult->rowCount() > 0;
    }
    
}