// Función para detectar si un elemento está en el viewport
function isElementInViewport(el, threshold = 0.1) {
    const rect = el.getBoundingClientRect();
    const windowHeight = window.innerHeight || document.documentElement.clientHeight;
    
    return (
        rect.top <= windowHeight * (1 - threshold) && 
        rect.bottom >= windowHeight * threshold
    );
}

// Función para manejar las animaciones de scroll
function handleScrollAnimations() {
    // Obtener todas las tarjetas que necesitan animación
    const animatedElements = document.querySelectorAll('.scroll-animate');
    
    animatedElements.forEach((element, index) => {
        if (isElementInViewport(element, 0.2)) {
            // Añadir un pequeño delay basado en el índice para efecto escalonado
            setTimeout(() => {
                element.classList.add('animate-in');
            }, index * 100);
        }
    });
}

// Función de debounce para optimizar el performance
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar animaciones al cargar la página
    handleScrollAnimations();
    
    // Manejar scroll con debounce para mejor performance
    const debouncedScrollHandler = debounce(handleScrollAnimations, 10);
    window.addEventListener('scroll', debouncedScrollHandler);
    
    // También ejecutar en resize por si cambia el viewport
    window.addEventListener('resize', debounce(handleScrollAnimations, 100));
});

// Intersection Observer API como alternativa más eficiente (navegadores modernos)
if ('IntersectionObserver' in window) {
    document.addEventListener('DOMContentLoaded', function() {
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -10% 0px'
        };
        
        const observer = new IntersectionObserver((entries) => {
            entries.forEach((entry, index) => {
                if (entry.isIntersecting) {
                    setTimeout(() => {
                        entry.target.classList.add('animate-in');
                    }, index * 100);
                    // Dejar de observar una vez que se anima
                    observer.unobserve(entry.target);
                }
            });
        }, observerOptions);
        
        // Observar todos los elementos con la clase scroll-animate
        const elementsToObserve = document.querySelectorAll('.scroll-animate');
        elementsToObserve.forEach(el => observer.observe(el));
    });
}